//
//  SCAdEnvironment.h
//  SCTVOSSDK
//
//  Copyright © 2018 smartclip. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <CoreGraphics/CGGeometry.h>

NS_ASSUME_NONNULL_BEGIN
@class SCAdMacros;

/**
 * This class is used to specify certain environment variables like desired bitrate, deviceType and screenSize
 */
@interface SCAdEnvironment : NSObject

///enum DeviceType specifies the device type
///on which this is running (must be set from outside)
typedef enum DeviceType {
    ///No device type specified
    deviceTypeNone = -1,

    ///Device type mobile
    deviceTypeMobile,

    ///Device type tablet
    deviceTypeTablet,

    ///Device type desktop
    deviceTypeDesktop,

    ///Device type tv
    deviceTypeTV
} DeviceType;

///enum NetworkReachability specifies the
///type of internet connection (must be set from outside)
typedef enum NetworkReachability {
    ///No reachability specified
    reachabilityNone = -1,

    ///Reachability wwan
    reachabilityWWAN,

    ///Reachability wifi
    reachabilityWIFI,

    ///Reachability unreachable
    reachabilityUnreachable
} NetworkReachability;

/// Timeout values that can be changed
typedef struct SCTimeouts {
    /// The number of seconds a single ad request can take to load
    /// before being terminated. Default is 1 seconds
    NSInteger adRequest;
    /// The total number of seconds a single ad request chain
    /// (VAST Wrappers and final VAST Inline tag) can take to load
    /// before being terminated. Default is 2 seconds.
    NSInteger adRequestChain;
    /// The total number of seconds an ad-slot request
    /// (from calling initAdSlot until the Promise resolves or rejects)
    /// can take before being terminated. Default is 4 seconds.
    NSInteger adSlotRequest;
    /// Representing the number of seconds a video creative can take to
    /// load before being terminated. Default is 3 seconds.
    NSInteger videoRequest;
    /// Representing the number of seconds a VPAID creative (JavaScript resource)
    /// can take to load before being terminated. Default is 1 seconds.
    NSInteger vpaidRequest;
    /// Representing the number of seconds a VPAID creative can take
    /// to initialize and respond (from calling VPAID.initAd() until
    /// the creative triggers the AdLoaded event) before being terminated. Default is 3 seconds.
    NSInteger vpaidReadiness;
    /// The maximum time in seconds a NonLinear ad is displayed
    /// (from triggering the impression) before being terminated.
    /// Note: The timer becomes paused for the time a creative has
    /// entered the linear state. Default is 10 seconds.
    NSInteger nonlinearMaxDuration;
} SCTimeouts;

/// Specify device type.
@property (nonatomic) DeviceType deviceType;

/// Specify network reachability
@property (nonatomic) NetworkReachability reachability;

/// Use a desiredBitrate, which overwrites bitrate calculation
@property (nonatomic) NSInteger desiredBitrate;

/// Value of the screenSize (if not specified, the values of the current device will be used)
@property (nonatomic) CGSize screenSize;

/// Desired mime types. Pass as array of strings
@property (nonatomic, strong) NSArray *desiredMimeTypes;

/*  In order to replace invalid ads in a pod of ads instantly at runtime, buffet ads from wrapper responses must be loaded up-front. prefetchBuffetAds determines whether stand-alone ads (buffet) from wrapper tags will be loaded together with the pod of ads or not. Defaulting to false if omitted.
 */
@property (nonatomic) BOOL prefetchBuffetAds;

/// Macros
@property (nonatomic, strong) SCAdMacros *macros;

/// Timeout values
@property (nonatomic) SCTimeouts timeouts;

/// Not available here
- (instancetype)init NS_UNAVAILABLE;

/// Default initializer
- (instancetype)initWithMacros:(nonnull SCAdMacros*)macros NS_DESIGNATED_INITIALIZER;
@end

NS_ASSUME_NONNULL_END
