//
//  SCAdSlotController.h
//  SmartclipSDKiOS
//
//  Copyright © 2018 smartclip. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
#import <SmartclipSDKiOS/SCAdPlayerEvent.h>
#import <SmartclipSDKiOS/SCPlayerEventListener.h>

@class SCAdInfo, SCAdConfiguration, SCPublicAdSlot, SCAdError, SCIcon;
@protocol SCOmidEventLoggingDelegate, SCAdListener;

NS_ASSUME_NONNULL_BEGIN

///Completionblock that delivers a String as result
typedef void (^SCStringResultCompletionBlock)(NSString* _Nullable result);
///Completionblock that delivers a String or an error as result
typedef void (^SCStringResultAndErrorCompletionBlock)(NSString* _Nullable result, NSError* _Nullable error);

///Completionblock without giving back a value
typedef void (^SCEmptyCompletionBlock)(void);

///The SCAdSlotController initializes and controls a single AdSlot. A valid SCAdSlotController must be inialized from the SCAdSessionController.
@interface SCAdSlotController : NSObject <SCPlayerEventListener>

///Readonly version of the SCAdConfiguration
@property (nonatomic, readonly) SCAdConfiguration *configuration;
///Register your adController as SCAdListener
@property (nonatomic, weak) id<SCAdListener> adListener;

/**
 * You start the playback of an ad slot with this function
 * - Parameter configutation: An instance of SCAdConfiguration
 */
- (void)startAdSlotWith:(SCAdConfiguration*)configuration;

/**
 * You start the playback of VMAP playlist with this function
 * - Parameter configutation: An instance of SCAdConfiguration
 */
- (void)startVMAPWith:(SCAdConfiguration*)configuration;

///Call this function to stop a VMAP playlist
- (void)stopAdPlaylist;

///Call this function if you want to stop and active adSlot (before it ends by itself)
- (void)stopAdSlot;

///Call this function if you want to skip an adSlot
- (void)skipAd;

///Cleanup after usage!
- (void)cleanup;

///Callback function for the video player that must be called for certain player events
- (void)playerEventCallbackWith:(SCAdPlayerEventType)playerEventType;

///Call this function for a more detailed adInfo
- (void)getAdInfo:(SCAdInfoCompletionBlock)completion;

///Call this function for a publicAdSlot object, that contains information about the current adSlot
- (void)getPublicAdSlot:(SCPublicAdSlotCompletionBlock)completion;

///Call this function for a publicAd object, that contains information about the current ad
- (void)getPublicAd:(SCPublicAdCompletionBlock)completion;

///Call this function when the user taps on the adView and clickthrough is active
- (void)onClickThrough;

///Call this function when the user taps on an industry icon
///If you want to display a dialog before the user clicks through, call this
///function from the "ok" alert action
- (void)onIconClickThrough:(SCIcon * _Nonnull)icon;

///Call this function when the industry icon switches to visible
- (void)onIconView:(SCIcon * _Nonnull)icon;

///Add your adView for visibility checks with this function
- (void)addVisibilityCheckerForView:(UIView*)adView;

///Register any view, that covers your adView but belongs to your ad playback views (like soundButton, replayButton etc.)
- (void)registerFriendlyViews:(NSMutableArray*)friendlyViews;

///Call this function if you want to pause playback
- (void)pausePlayback;

///Call this function if you want to start playback
- (void)startPlayback;

///Call this function if you want to resume playback
- (void)resumePlayback;

///Add views that belong to the player (like soundbuttons etc.) and should not
///be registered as views that cover the adView
- (void)setFriendlyViews:(NSMutableArray*)friendlyViews;

///Call this function, if while streaming an ad the player stops due to an empty buffer
- (void)onBufferEmpty;

///Call this function, if the player can continue after the buffer was empty and is
///able to play on now
- (void)onBufferRefilled;

///Call this function for a detailed adError error description
- (void)getAdError:(SCAdErrorCompletionBlock)completion;

/// Can be used to set a receiver for omid specific event logging
- (void)setOmidEventLoggingDelegate:(id<SCOmidEventLoggingDelegate>)omidEventLogger;

///Call this function if the loading of the adTag succeeded
- (void)loadAdSuccess;

///Call this function if the loading of the adTag failed
- (void)loadAdFailure;

///Call this function to unmute the adPlayer
- (void)unmutePlayer;

///Call this function to mute the adPlayer
- (void)mutePlayer;

///Call this function to check if the player currently is muted
- (BOOL)isMuted;

///Please be aware that calling this function results in sending
///a click tracker! only use it in companion with a customizableClickThroughAlert,
///other implementations of clickThrough handling already send a click tracker,
///so calling this function in that situation results in multiple click trackers being sent.
- (void)clickTroughURL:(SCStringResultCompletionBlock)completion;

///Call this function when your video player has changed its size
- (void)resize;

///This is a generic function, that can return any core property
///Please use this function only, if you need a core value, that you cannot
///address otherwise by using SmartclipSDKiOS
///You cannot use this function to call any java script function on the smartclientcore!
- (void)getSmartclientcoreValue:(NSString* _Nonnull)coreProperty completion:(SCStringResultAndErrorCompletionBlock)completion;
@end

NS_ASSUME_NONNULL_END
