var PLAYER_FACTORY_VERSION = '3.2';
// defines empty functions that get filled with the load-Promise callbacks
var loadRequestPromiseResolve = function() {};
var loadRequestPromiseReject = function() {};

function FacadeBase() {
    this.handshakeVersion = function _(apiFacadeVersion) {
        return PLAYER_FACTORY_VERSION;
    };
    /**
     * @param {string} url Media file URI
     * @param {Boolean} disableSeeking Wether the requested source can be seeked back and forth
     * @returns {Promise} Either resolves if the player accepted the play request or not
     */
    this.load = function _(url, disableSeeking = false) {
        return new Promise(function(resolve, reject) {
            loadRequestPromiseResolve = resolve;
            loadRequestPromiseReject = reject;
            var arguments = {'url': url, 'disableSeeking': disableSeeking};
            var accept = callNativeFunction("loadAdWithUrlString:disableSeeking:", arguments);
            // the native code should then respond by calling either
            //  callJSFunction("loadRequestPromiseResolve")
            //  callJSFunction("loadRequestPromiseReject")
            // depending on whether the MediaFile could be loaded (resolve) or not (reject).
        });
    };

    /**
     * Sets up a function that will be called whenever the specified event is delivered to the video player.
     * @param {string} type - A case-sensitive string representing the player event type to listen for.
     * @param {Function} callback - The function which receives a notification when an event of the specified type occurs.
     */
    this.addEventListener = function _(event, callback) {
        let eventDispatcher = getEventDispatcher();
        eventDispatcher.addEventListener(event, callback);
    };

    /**
     * Removes an event listener from the video player that has been registered with `addEventListener()`
     * _*Note:* The event listener to be removed must match with the listener that has been added._
     * @param {string} type - A case-sensitive string representing the player event type to listen for.
     * @param {Function} callback - The function which receives a notification when an event of the specified type occurs.
     */
    this.removeEventListener = function _(event, callback) {
        let eventDispatcher = getEventDispatcher();
        eventDispatcher.removeEventListener(event, callback);
    };


    /**
     * This function is an iOS specific function and must survive FacadeBase updates!
     * It is called for every core event and dispatches those events to the mobile player
     */
    this.onEventCallback = function _(event) {
        var message = {'name':"onEventCallback:", 'event': event};
        postPlayerFacadeMessage(message);
    };

    /**
     * Requests the video player to return to the content and last known playhead position.
     * @returns {Promise} Resolves if the player accepted the request
     */
    this.lockContent = function _() {
        var message = {'name':"lockContent"};
        postPlayerFacadeMessage(message);
        return Promise.resolve(true);
    };

    /**
     * Requests the video player to return to the content and last known playhead position.
     * @returns {Promise} Resolves if the player accepted the request
     */
    this.releaseContent = function _() {
        let accept = callNativeFunction("releaseContent", null);
        return Promise.resolve(accept);
    };

    /**
     * @returns {SourceObject} Current content source and playhead progress
     */
    this.getCurrentContentSource = function _() {
        return callNativeFunction("getCurrentContentSource", null);
    };

    /**
     * Requests the current playhead position from the video player.
     * @returns {Number} Current time in seconds.
     */
    this.getCurrentTime = function _() {
        let currentTime = callNativeFunction("getCurrentTime", null);
        return parseFloat(currentTime);
    };

    /**
     * Requests the content-duration from the video player.
     * @returns {Number} Total time in seconds.
     */
    this.getDuration = function _() {
        let duration = callNativeFunction("getDuration", null);
        return parseFloat(duration);
    };

    /**
     * Requests the latest status of the video player, site, and other external factors.
     * @param {string} adBreakType - Linearity of the upcoming ad slot. See [`AdBreak.type`](setup-sequential-config#adbreak+type)
     * @returns {EnvironmentVars} Description object of the most recent error or `null`.
     */
    this.getEnvironmentVars = function _(adBreakType) {
        var arguments = {'adBreakType': adBreakType};
        let environmentVars = callNativeFunction("getEnvironmentVars:", arguments);
        return JSON.parse(environmentVars);
    };

    /**
     * Requests details about the most recent error that has occurred on the video player.
     * @returns {MediaError} Description object of the most recent error or `null`.
     */
    this.getError = function _() {
        let error = callNativeFunction("getError", null);
        return JSON.parse(error);
    };

    /**
     * Requests the current size and viewMode from the video player.
     * @returns {ElementSize} Current player dimension specs
     */
    this.getPlayerSize = function _() {
        let playerSize = callNativeFunction("getPlayerSize", null);
        return JSON.parse(playerSize);
    };

    /**
     * Requests the current size and viewMode from the viewport that surrounds the video player.
     * @returns {ElementSize} The dimension specs of the element
     */
    this.getViewportSize = function _() {
        let viewPortsize = callNativeFunction("getViewportSize", null);
        return JSON.parse(viewPortsize);
    };

    /**
     * Requests the video player element
     * @returns {Element} The video element
     * @deprecated
     */
    this.getVideoElement = function _() {
    };

    /**
     * Requests the player element
     * @returns {Element} The video or audio element
     */
    this.getPlayerElement = function _() {
    };

    /**
     * Requests the video player to change the volume.
     * @param {Number} volume - Requested volume level between 0-1
     * @returns {Promise} Resolves if the video player accepted the volume change request
     */
    this.setVolume = function _(volume) {
        var message = {'name':"setVolume:", 'volume': volume};
        postPlayerFacadeMessage(message);
    };

    /**
     * Requests the current volume level from the video player.
     * @returns {Number} The current volume level between 0-1
     */
    this.getVolume = function _() {
        let volume = callNativeFunction("getVolume", null);
        return parseFloat(volume);
    };

    /**
     * Requests the current muted state from the video player.
     * @returns {Boolean} Muted state
     */
    this.isMuted = function _() {
        return callNativeFunction("isMuted", null);
    };

    /**
     * Requests the video player to mute playback.
     * @returns {Promise} Resolves if the video player has been muted
     */
    this.mute = function _() {
        postPlayerFacadeMessage("mute");
    };

    /**
     * Requests the video player to unmute playback.
     * @returns {Promise} Resolves if the video player has been unmuted
     */
    this.unmute = function _() {
        postPlayerFacadeMessage("unmute");
    };

    /**
     * Requests the current play state from the video player.
     * @returns {Boolean} Playback state
     */
    this.isPaused = function _() {
        return callNativeFunction("isPaused", null);
    };

    /**
     * Requests the video player to pause playback.
     * @returns {Promise} Resolves if the video player has been paused
     */
    this.pause = function _() {
        postPlayerFacadeMessage("pausePlayback");
    };

    /**
     * Requests the video player to start (resume) playback.
     * @returns {Promise} Resolves if the video player has been resumed
     */
    this.play = function _() {
        postPlayerFacadeMessage("startPlayback");
    };

    /**
     * Requests the video player to change the playhead position to the specified time.
     * @param {Number} offset - Requested playhead position
     * @returns {Promise} Resolves if the video player has finished seeking to the specified position
     */
    this.seek = function _(offset) {
        var message = {'name':"seek:", 'offset': offset};
        postPlayerFacadeMessage(message);
    };
}
