[> Documentation-Index](index.md)

# smartclientcore - Static API

The **static API** can be used to create and connect instances of the smartclientcore with the video player.
It is available from the global namespace on the `window` object.


## Table of contents
- [Methods](#methods)
- [Properties](#properties)
  
## Methods
### connect(player : PlayerFacade) : smartclientcore
Sets up a new smartclientcore instance for the given video player.

### setup(pluginSetup : [GlobalConfig](setup-global-config.md)) : Promise<[AdSlotAPI](AdSlotAPI.md)>  
Configures the smartclientcore instance with the given configuration, as specified in [Global Configuration](setup-global-config.md).
Returns a promise that either resolves with the [AdSlotAPI](AdSlotAPI.md) 
or rejects with one of the following [PluginError](AdSlotAPI.md#PluginError):

#### Setup Error - Codes Table
| Code | Description |
| ---- | ----------- |
| 60   | Invalid Setup - The specified `pluginSetup` does not match with the [GlobalConfig](setup-global-config.md) definition. |
| 61   | Player API not available - The PlayerFacade has not been connected before starting the setup process. |
| 62   | Ad Reinsertion Penalty - AdReinsertion initialization failed, either due to a invalid [configuration](setup-global-config.md#adreinsertion) or issues that must be answered by the AdReinsertion vendor. |
| 63   | Ad Reinsertion Initialization Timeout - AdReinsertion initialization took too long, either caused by the [timeout limitation](setup-global-config.md#timeouts) or issues that must be answered by the AdReinsertion vendor. |
| 64   | Ad Reinsertion usage declined by the vendor. |


#### How to catch Setup Errors
```javascript
let adSlotAPI = null;
let playerFacade = new PlayerFacade();

// We won't create the entire configuration from scratch, we'll use the default object...
let globalConfig = smartclientcore.SCHEMA.create(smartclientcore.SCHEMA.GLOBAL_CONFIG);

// ... and only adjust selected properties.
globalConfig.debug = true;
globalConfig.adReinsertion = {
	homad: {
		enabled: true,
		setup: {
			globalConfig: 'https://example.com/global_config.json',
			clientConfig: 'https://example.com/client_config.json',
			enableSessionCoverage: false,
			activationCallback: (_reason) => {
				return new Promise((resolve, reject) => {
					
					// The reason for activating HOMAD sounds reasonable?
					resolve();
					
					// otherwise
					//reject();
				});
			}
		}
	}
};

// Requests the AdSlotAPI for the given configuration or fails with reason.
smartclientcore.connect(playerFacade).setup(globalConfig).then((_adSlotAPI) => {
	adSlotAPI = _adSlotAPI; 
	smartclientcore.DEBUG.info('Successfully created publisher plugin instance', this.adSlotAPI);
}, (failure) => {
	smartclientcore.DEBUG.warn('Error while creating publisher Advertising Module instance.', failure);
	
	switch(failure.code) {
		case 60:
			const isValidConfig = smartclientcore.SCHEMA.validate(this.globalConfig, smartclientcore.SCHEMA.GLOBAL_CONFIG);
			smartclientcore.DEBUG.info('[publisherModule] Schema validation failed?', isValidConfig);
			break;
		case 61:
			smartclientcore.DEBUG.info('[publisherModule] Player API not available. Did you call `smartclientcore.connect(withYourPlayerFacade)` before?');
			break;
		case 62:
			smartclientcore.DEBUG.info('[publisherModule] AdReinsertion initialization failed.');
			
			/** If the issue appears during development
			 *  - double check the `adReinsertion` config for any typos or wrong URLs
			 *  - watch out for missing sources in the network tab of the developer tools
			 *    and contact the source owner
			 *  - deactivate the ad-blocker
			 */
			
			/** Otherwise you are facing a very bad ad-blocker.
			 *  - AdReinsertion will not work
			 *  - force the user to deactivate the ad-blocker
			 *  - or play out the content without ads
			 */
			break;
		default:
			smartclientcore.DEBUG.info('[publisherModule] Unknown setup error.', failure);
			break;
	}
});
```
<sub>Sample code :: How to catch setup errors.</sub>

## Properties
### AD_TYPE
Declaration of ad types as ENUMs that can be used with requesting an ad slot.

### EVENT: Object
Declaration of events as ENUMs that occur during playback of a single ad or mutliple ads delivery.
See [smartclientcore - Ad Events](events.md) for more details.

### EVENTS: Array
List of event names as defined in the EVENT object.

### SCHEMA: Object
The SCHEMA Object introduces default schemes for frequently used Objects and methods that help creating and validating these.
All SCHEMA definitions and helpers are accessible via the `smartclientcore.SCHEMA` Object.

#### Schemes
**AD_BREAK : JSONSchema**  
Setup definition for a commercial break. Properties are described within section [Sequential Config: Ad-Break](setup-sequential-config.md#ad-break-adbreak).

**GLOBAL_CONFIG : JSONSchema**  
Setup definition for the current session that is applied to all commercial breaks.
 Properties are described within section [Global Config](setup-global-config.md).

#### Helpers
**validate(input : Object, schema : JSONSchema) : Boolean**  
Can be used to check the validity of Configuration Objects against a corresponding schema.

**create(schema : JSONSchema) : Object**  
Returns a qualified Object for the given schema definition, filled with the expected default values.
 
### supportedFacadeVersion: String
Returns the version number for the supported Facade interface.

### error: [PluginError](AdSlotAPI.md#PluginError)
Requests details about the most recent error that has occurred.
